#include <UAW API.h>



enum TerrainID : UAW_Extension_TerrainID {
	sampleTerrain = 123, // choose any number you like
};

enum ScenarioID : UAW_Extension_ScenarioID {
	sampleScenario = 123, // choose any number you like
};

constexpr auto tileSize = 2048.f; // ~2 km – good speed/space tradeoff

enum StringID : UAW_Extension_StringID {
	extensionName,
	extensionAuthor,
	extensionUrl,
	extensionDescription,
	sampleScenarioName,
	sampleScenarioDescription,
};

// Returns a string in UAW’s current language.
UAW_Char const * UAW_CALL stringFromStringID(
	UAW_Extension const *  extension, // explained later
	UAW_Extension_StringID id
) {
	switch(id) {
		case extensionName:
			return u"Sample Extension";
		case extensionAuthor:
			return u"Someone";
		case extensionUrl:
			return u"http://example.com";
		case extensionDescription:
			return u"Demonstrates the implementation of a basic UAW extension.";
		case sampleScenarioName:
			return u"Sample Scenario";
		case sampleScenarioDescription:
			return u"Demonstrates the implementation of a basic UAW scenario.";
		default:
			return u"YOU FORGOT TO DEFINE THIS STRING!";
	}
}

struct UAW_Extension_Terrain {
	// empty placeholder
};

int UAW_CALL populateScenarioWithTerrain(
	UAW_Extension_Terrain const * terrain,
	UAW_API_ScenarioPopulate *    uaw
) {
	return 1; // nothing to do yet
}

void UAW_CALL unloadTerrain(
	UAW_Extension_Terrain * terrain
) try {
	delete terrain;
} catch(...) {
}

struct UAW_Extension_TerrainTile {
	unsigned int indexN;
	unsigned int indexE;
};

// Load extension-specific data for the given tile.
//  • return “NULL” on failure or if there is no tile at this point
UAW_Extension_TerrainTile * UAW_CALL loadTerrainTileAt(
	UAW_Extension_Terrain *     terrain,
	unsigned int                indexNorth,
	unsigned int                indexEast,
	UAW_TerrainTileProperties * properties
) try {
	return new UAW_Extension_TerrainTile{ indexNorth, indexEast };
} catch(...) {
	return nullptr;
}

// Release extension-specific data for the tile.
void UAW_CALL releaseTerrainTile(
	UAW_Extension_Terrain *     terrain,
	UAW_Extension_TerrainTile * tile
) try {
	delete tile;
} catch(...) {
}

struct UAW_Extension_TerrainView {
	// empty placeholder
};

UAW_Extension_TerrainView * UAW_CALL onTerrainViewCreate(
	UAW_Extension_Terrain * terrain,
	int                     tileIndexNorth,
	int                     tileIndexEast,
	int                     rangeInTiles
) try {
	return new UAW_Extension_TerrainView;
} catch(...) {
	return nullptr;
}

void UAW_CALL onTerrainViewChange(
	UAW_Extension_Terrain *     terrain,
	UAW_Extension_TerrainView * view,
	UAW_API_TerrainGraphics *   uaw,
	int                         tileIndexNorth,
	int                         tileIndexEast,
	int                         rangeInTiles
) {
	// nothing to do yet
}

void UAW_CALL onTerrainViewDestroy(
	UAW_Extension_Terrain *     terrain,
	UAW_Extension_TerrainView * view
) try {
	delete view;
} catch(...) {
}

void UAW_CALL renderTerrain(
	UAW_Extension_Terrain const *    ,
	UAW_Extension_TerrainView const *,
	UAW_Matrix_3x3_float             worldToEye_raw,
	UAW_NED_float                    eyeInTile,
	float                            zoomScale,
	unsigned int                     widthInAnyUnit,
	unsigned int                     heightInAnyUnit,
	UAW_Clock                        utc
) {
	// nothing to do yet
}

struct UAW_Extension_TerrainMapView {
	// empty placeholder
};


// Reserves all data for rendering an overview map of the terrain to a virtual screen.
//  • returns “NULL” on failure
//  • “destroyMapView()” will be called after use
UAW_Extension_TerrainMapView * UAW_CALL createTerrainMapView(
	UAW_Extension_Terrain * terrain,
	UAW_API_StartScreens *  uaw
) try {
	return new UAW_Extension_TerrainMapView;
} catch(...) {
	return nullptr;
}

// Draws an overview map of the terrain to a virtual screen.
//  • “range” is the radius of the mapped area, in meters
//  • “transformation” is never “NULL” and transforms the map from meters to pixels on the screen
//    – 1ˢᵗ column TODO
//    – 2ⁿᵈ column TODO
//    – 3ʳᵈ column TODO
void UAW_CALL drawTerrainMap(
	UAW_Extension_Terrain const *        terrain,
	UAW_API_Screen *                     uaw,
	UAW_Extension_TerrainMapView const * view,
	UAW_NED_double const *               center,
	float                                range,
	UAW_Matrix_3x2_float const *         transformation
) {
	// nothing to do yet
}

// Releases all data for rendering this overview map of the terrain.
void UAW_CALL destroyTerrainMapView(
	UAW_Extension_Terrain *        terrain,
	UAW_API_StopScreens *          uaw,
	UAW_Extension_TerrainMapView * view
) try {
	delete view;
} catch(...) {
}


// Returns the party that owns the territory at the given point, or “nullptr” if it is neutral.
UAW_Scn_Party * UAW_CALL terrainTerritoryAt(
	UAW_Extension_Terrain const *    ,
	UAW_Extension_TerrainTile const *,
	UAW_NE_float                     relativeToTileCenter
) {
	return nullptr; // neutral
}

// Computes the collision of a line segment with a terrain tile.
//  • origin and direction are meters relative to tile center
UAW_TerrainRayHit UAW_CALL rayVsTerrainTile(
	UAW_Extension_Terrain const *    ,
	UAW_Extension_TerrainTile const *,
	UAW_NED_float                    origin,
	UAW_NED_float                    direction
) {
	return { 2.f, { }, }; // relative distance >1 means “no collision”
}

// For a specific point in the terrain, computes the elevation above mean sea level (MSL), in meters, as well as the normal.
//  • elevation can be negative for some places on Earth
UAW_Lvl_ElevationAndNormal UAW_CALL terrainElevationAt(
	UAW_Extension_Terrain const *     terrain,
	UAW_Extension_TerrainTile const * tile,
	UAW_NE_float                      position
) {
	return { 0.f, { 0.f, 0.f, -1.f } };
}


UAW_Extension_Terrain * UAW_CALL loadTerrain(
	UAW_Extension *             extension,
	UAW_API_TerrainLoad *       uaw,
	UAW_Extension_TerrainID     id,
	UAW_Extension_Terrain_API * result
) try {
	result->unload           = &unloadTerrain;
	result->populateScenario = &populateScenarioWithTerrain;
	result->loadTileAt       = &loadTerrainTileAt;
	result->releaseTile      = &releaseTerrainTile;
	result->onViewCreate     = &onTerrainViewCreate;
	result->onViewChange     = &onTerrainViewChange;
	result->onViewDestroy    = &onTerrainViewDestroy;
	result->render           = &renderTerrain;
	result->territoryOf      = &terrainTerritoryAt;
	result->rayVsTerrainTile = &rayVsTerrainTile;
	result->elevationAt      = &terrainElevationAt;
	result->createMapView    = &createTerrainMapView;
	result->drawMap          = &drawTerrainMap;
	result->destroyMapView   = &destroyTerrainMapView;
	return new UAW_Extension_Terrain;
} catch(...) {
	return nullptr;
}

struct UAW_Extension_Scenario {
	// empty placeholder
};

void UAW_CALL destroyScenario(
	UAW_Extension_Scenario * scenario
) try {
	delete scenario;
} catch(...) {
}

void UAW_CALL prepareScenarioRendering(
	UAW_Extension_Scenario * scenario,
	UAW_Clock                utc
) {
	// nothing to do yet
}

UAW_Extension_Scenario * UAW_CALL createScenario(
	UAW_Extension *              extension,
	UAW_API_ScenarioPopulate *   uaw,
	UAW_Extension_ScenarioID     id,
	UAW_Extension_Scenario_API * methods
) try {
	// Pass our callbacks:
	methods->destroy          = &destroyScenario;
	methods->prepareRendering = &prepareScenarioRendering;
	return new UAW_Extension_Scenario;
} catch(...) {
	return nullptr;
}

extern "C" __declspec(dllexport) int UAW_CALL UAW_EXTENSION_CREATE(
	UAW_API_Extension *       uaw,
	UAW_Extension_Callbacks * result,
	void *                    legacyPleaseIgnore
) {
	// If you work object-oriented, create an instance and store its address here.
	// It will be passed back to you with every callback.
	result->implementation = nullptr;

	// If you need a chance to clean up before UAW quits, place your callback here.
	result->unload = nullptr;

	// For display in the Extensions menu (from the Strings chapter sample):
	result->string      = &stringFromStringID;
	result->name        = extensionName;
	result->author      = extensionAuthor;
	result->url         = extensionUrl;
	result->description = extensionDescription;

	// Declare sample terrain:
	UAW_TerrainSpecs terrain = { };
	terrain.id             = sampleTerrain;
	terrain.latitude       = 49.f; // origin is somewhere in the Celtic Sea
	terrain.longitude      = -10.f;
	terrain.tileSidelength = tileSize;
	terrain.load           = &loadTerrain; // pass our callback
	uaw->newTerrain(terrain);

	// Declare an empty scenario using the sample terrain:
	UAW_ScenarioSpecs scenario = { };
	scenario.id        = sampleScenario;
	scenario.terrainID = sampleTerrain;
	scenario.name      = sampleScenarioName;
	scenario.summary   = sampleScenarioDescription;
	scenario.create    = &createScenario; // pass our callback
	// Do not spawn at the edge of the scenario but a few kilometers into it:
	scenario.spawnPoint.north = 12'000.;
	scenario.spawnPoint.east  = 12'000.;
	scenario.spawnPoint.down  = -1'000.;
	scenario.spawnHeading     = 45; // degrees
	uaw->newScenario(scenario);

	return 1; // success (zero for failure)
}